#!/bin/ksh
# This script is assumed:
#   a) to be run by the Korn ("ksh") shell, and
#   b) to reside in the same directory as the files to be installed.

# Prototype Balaur firmware installer script 
# intended to run on the target Balaur SmartCard4 
# derived from a LC-QQQ Smartcard4 including Slackware Linux.
#
# Usage:  this_script $OPT1 $RETAINEXISTING > this_script.log  2>&1
#
# where:  $OPT1 = SIM specifies to install simulator versions
#                 RUN specifies to install real instrument versions
#         $RETAINEXISTING = YES specifies to retain existing values in 
#                               non-volatile storage of specific parameters.
#
# MODIFICATION HISTORY:
# 26May2008 KK  (Knute Kresie) composed the initial version.
#           Derived partly from do_tarball.ksh
#           FUTURE:  Add login as "root", set SC4 system "date", reboot.
# 03JUL2008 KK  Changes to remove qqq_old/ if it exists, before proceeding.
#           then copy qqq_old/* to qqq/* to preserve some files.
#           Use dot.profile file name instead of .profile name to work-around
#           Microcrap Windows problems with file names starting with ".".
# 06JUL2008 KK  Made changes to handle case when no qqq/ directory exists yet.
# 06Dec2008 KK  Added "rm -f $DEST_PATH/$FILE" to prevent problems due to
#           unexpected read-only file permissions when installing their
#           sucessors .
# 11Mar2009 KK  Added installation of $HOME/PlxCm/* files to enable
#           customer engineers to update MainBoard FPGA code in the field.
#           The PlxCm files and install procedure are supplied by R. Roushall.
#           Added calling the retain_parms program to optionally retain 
#           (preserve) specific existing parameter values
#           (Quad Polarity, Quad Frequency) that were set at the factory or
#           in the field by a customer engineer. See RETAINEXISTING.
# 13Mar2009 KK  Changed name of retain_parms to retain_values.
#           Changed name of PlxCm directory to PlxCmDir to avoid confusion
#           with PlxCm program name under that directory.
#

  set -u            # treat undefined parameters as error
  ID=`basename $0`

#---------------------- Define Functions ----------------------------------
   function tmsg  { print -u2 "`date "+%Y%m%d.%H%M%S"` ${ID}: $1"; }
   function pmsg  { print -u1 "`date "+%Y%m%d.%H%M%S"` ${ID}: $1"; }
   function check_status { STATUS=$1 ; LOC=$2
                           if (($STATUS != 0))
                           then
                             tmsg "ABORT with STATUS=$STATUS at $LOC"
                             exit $STATUS
                           else
                             tmsg "STATUS=$STATUS at $LOC"
                           fi
                          }

#--------------------------------------------------------------------------


  tmsg "Starting."
  tmsg "@(#)\$Id: installFW.ksh, 13Mar2009 16:24 $"

  OPT1=$1

  if [[ $OPT1 != "SIM" ]] && [[ $OPT1 != "RUN" ]]
  then
    tmsg "Invalid InstallationType = $OPT1"
    exit 1
  fi

  RETAINEXISTING=$2

  tmsg "InstallationType=$OPT1  RetainExistingParms=$RETAINEXISTING"
  myPWD=`pwd`
  tmsg "PWD=$myPWD"


  DEST_PATH=$HOME     # typical $HOME is "/home/sc4"
#  DEST_PATH=$HOME/KKtest  # for TESTING this script

  tmsg "DEST_PATH=$DEST_PATH"

  if [[ -d ${DEST_PATH}/qqq ]]
  then
    tmsg "Found ${DEST_PATH}/qqq"
  else
    # Do this to simplify script logic below.
    mkdir ${DEST_PATH}/qqq
    pmsg "No qqq/ directory found initially." > ${DEST_PATH}/qqq/initial_mkdir
    check_status $? made_initial_qqq_dir
  fi

# Remove the qqq_old/ directory, if it exists, to enable
# that name to be used later below.

  if [[ -d ${DEST_PATH}/qqq_old ]]
  then
    tmsg "Removing existing $DEST_PATH/qqq_old"
    rm -rf ${DEST_PATH}/qqq_old
    check_status $? removed_qqq_old
  fi


# The following renaming of qqq/ to qqq_old/ enables installing a new version
# of qqq/* (later, below) without needing to shut-down the the qqqServer
# process that was started from qqq_old/qqqServer.  This is for operational 
# convenience/simplicity.
# This should work OK assuming qqqServer is not opening a file using
# an absolute path while this script is running.

  if [[ -d ${DEST_PATH}/qqq ]]
  then
    tmsg "Renaming (mv) $DEST_PATH/qqq to $DEST_PATH/qqq_old"
    mv ${DEST_PATH}/qqq ${DEST_PATH}/qqq_old
    check_status $? renamed_qqq_TO_qqq_old
  fi

# Do updates of $HOME

  /sbin/ifconfig | grep inet | read junk1 ipaddr junk3
  date > ${DEST_PATH}/THIS_IS_IP$ipaddr   # create a convenient SC4 ID file
  check_status $? fetched_$ipaddr   

  FILE=.profile
  rm -f $DEST_PATH/$FILE
  cp -p dot$FILE $DEST_PATH/$FILE  # .profile name is difficult for MSWindows
  check_status $? installed_$FILE

  FILE=shellVars
  rm -f $DEST_PATH/$FILE
  cp -p $FILE ${DEST_PATH}/$FILE
  check_status $? installed_$FILE   
  chmod 755 ${DEST_PATH}/$FILE     # make executable by /etc/rc.d/rc.local

  FILE=showstr
  rm -f $DEST_PATH/$FILE
  cp -p $FILE ${DEST_PATH}/$FILE
  check_status $? installed_$FILE   
  chmod 755 ${DEST_PATH}/$FILE     # make executable

# Create a new qqq/ and Copy qqq_old/ to it. 

  mkdir ${DEST_PATH}/qqq
  check_status $? mkdir_qqq
# The following copy qqq_old/ to qqq/ is done to preserve various files
# the qqqServer may have created (e.g. MS1QuadFreq.txt, FaultLog.txt, ...)
# and enable the firmware to be updated in the new qqq/ while qqqServer
# is running from qqq_old/ .
  cp -p ${DEST_PATH}/qqq_old/* ${DEST_PATH}/qqq
  check_status $? copy_qqq_old_TO_qqq

# Update the newly-created qqq/ directory

  DEST_PATH=${DEST_PATH}/qqq        # to reduce some typing
  tmsg "DEST_PATH=$DEST_PATH"   # KKDEBUG

  FILE=QQQDefaultParms.txt
  rm -f $DEST_PATH/$FILE
  cp -p ${FILE} $DEST_PATH/$FILE
  check_status $? installed_${FILE} 

  OLDFILE=${HOME}/qqq_old/QQQDefaultParms.txt
  NEWFILE=$DEST_PATH/$FILE
  if [[ $RETAINEXISTING != "NO" ]] && [[ -f $OLDFILE ]]
  then
    # Run program to preserve existing settings for
    # Quad Polarity, Quad Frequency
    # in QQQDefaultParms.txt file, according to user choice.
    # Make the fail-safe action be to retain existing values
    # that were set at the factory or by customer engineer.
    retain_values $OLDFILE $NEWFILE
    check_status $? do_retain_existing_parms 
    tmsg "Retained existing values for specific parameters."
    tmsg "(Quad Polarity, Quad Frequency)"
  else
    tmsg "Installed default values for all parameters."
  fi
  check_status $? default_parms_processing

  FILE=qqqServer
  rm -f $DEST_PATH/$FILE
  cp -p ${OPT1}/${FILE} $DEST_PATH/$FILE
  check_status $? installed_${OPT1}/${FILE} 
  chmod 755 $DEST_PATH/$FILE    # make executable by /etc/rc.d/rc.local

  FILE=QQQ_DSP_FW.ldr
  rm -f $DEST_PATH/$FILE
  cp -p ${OPT1}/${FILE} $DEST_PATH/$FILE
  check_status $? installed_${OPT1}/${FILE} 

# Install $HOME/PlxCmDir/* directory and files

  if [[ -d ${HOME}/PlxCmDir ]]
  then
    tmsg "Found ${HOME}/PlxCmDir"
  else
    mkdir ${HOME}/PlxCmDir
    pmsg "Created directory ${HOME}/PlxCmDir"
    check_status $? check_PlxCmDir
  fi

  FILE=PlxCmDir/PlxCm
  if [[ -f ${HOME}/$FILE ]]
  then
    tmsg "Removing existing ${HOME}/$FILE"  # to be sure it can be replaced
    rm -f ${HOME}/$FILE
  fi
  check_status $? check_existing_PlxCm_file

  # Need to "rm -f PlxCmDir/*.xsvf" files first?  Without error checking?
  # Assume PlxCmDir/*.xsvf files will =not= be write-protected.
  # Next copy new PlxCmDir/* files (PlxCm, *.xsvf) to target directory

  cp -p PlxCmDir/* ${HOME}/PlxCmDir
  check_status $? install_PlxCmDir_files


  tmsg "Done"
